import os
import json
import configparser

project_root = os.path.dirname(__file__)

def read_config(local_config_path, default_config_path):
    config = configparser.ConfigParser()

    if os.path.exists(local_config_path):
        config.read(local_config_path)
        print(f'Loaded configuration from {local_config_path}')
    else:
        config.read(default_config_path)
        print(f'Loaded configuration from {default_config_path}')

    return config

test_config = read_config(
    os.path.join(project_root, 'local_config.ini'),
    os.path.join(project_root, 'default_config.ini'),
)

llm_type = test_config.get('EXP', 'llm')

dataset = test_config.get('EXP', 'dataset')
output_dir = os.path.join(project_root, 'output', llm_type, dataset)

def calculate_success_rate(result_files, output_dir):
    results = {}
    
    for result_file in result_files:
        filename = os.path.join(output_dir, f"{result_file}_results.jsonl")
        
        total_cases = 0
        success_cases = 0
        
        with open(filename, 'r', encoding='utf-8') as f:
            for line in f:
                try:
                    data = json.loads(line.strip())
                    total_cases += 1
                    if data['judge']['success_status']:
                        success_cases += 1
                except json.JSONDecodeError:
                    print(f"Warning: Invalid JSON in {result_file}")
                except KeyError:
                    print(f"Warning: Missing required fields in {result_file}")
        
        success_rate = (success_cases / total_cases * 100) if total_cases > 0 else 0
        results[result_file] = {
            'total_cases': total_cases,
            'success_cases': success_cases,
            'success_rate': round(success_rate, 2)
        }
    
    print("\nAttack Success Rate Summary:")
    print("-" * 50)
    print(f"{'Method':<15} {'Success Rate':<12} {'Success/Total'}")
    print("-" * 50)
    for method, stats in results.items():
        print(f"{method:<15} {stats['success_rate']:>6.2f}%     {stats['success_cases']}/{stats['total_cases']}")
    
    return results

result_files = [
    "baseline", 
    "dra", 
    "art_prompt", 
    "flip_attack", 
    "pair", 
    "sata", 
    "pass"
]
calculate_success_rate(result_files, output_dir)